<?php
/**
 * Fingerprint Collector - Bot Intelligence Gatherer
 * 
 * Bu dosya tüm gelen isteklerin detaylarını toplar ve analiz için saklar.
 * .htaccess ile auto_prepend_file olarak ayarlandığında her isteği yakalar.
 * 
 * Kurulum:
 * 1. Bu dosyayı ve diğer dosyaları web root'a kopyala
 * 2. .htaccess dosyasını web root'a kopyala
 * 3. data/ klasörüne yazma izni ver (chmod 755)
 * 4. Verileri görmek için: yourdomain.com/_viewer.php?key=YOUR_SECRET_KEY
 */

// ============== CONFIGURATION ==============
define('COLLECTOR_SECRET_KEY', 'change_this_secret_key_123'); // Viewer erişim anahtarı
define('DATA_DIR', __DIR__ . '/data');
define('DB_FILE', DATA_DIR . '/fingerprints.db');
define('MAX_RAW_SAMPLES', 500); // Son N raw request sample'ı sakla
define('ENABLE_TIMING_ANALYSIS', true); // Request timing analizi
// ============================================

// Data dizinini oluştur
if (!file_exists(DATA_DIR)) {
    mkdir(DATA_DIR, 0755, true);
    // .htaccess ile data klasörünü koru
    file_put_contents(DATA_DIR . '/.htaccess', "Deny from all\n");
}

// Collector dosyalarına erişimi engelle
$requestUri = $_SERVER['REQUEST_URI'] ?? '';
if (strpos($requestUri, '_collector.php') !== false || 
    strpos($requestUri, '_viewer.php') !== false ||
    strpos($requestUri, '/data/') !== false) {
    return; // Bu dosyalara doğrudan erişimi atla
}

// SQLite bağlantısı
try {
    $db = new PDO('sqlite:' . DB_FILE);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Tabloları oluştur (yoksa)
    initDatabase($db);
    
    // İsteği kaydet
    collectRequest($db);
    
} catch (Exception $e) {
    // Sessizce hata logla, kullanıcıya gösterme
    error_log("Fingerprint Collector Error: " . $e->getMessage());
}

/**
 * Veritabanı tablolarını oluştur
 */
function initDatabase($db) {
    // User-Agent istatistikleri
    $db->exec("CREATE TABLE IF NOT EXISTS user_agents (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_agent TEXT UNIQUE,
        count INTEGER DEFAULT 1,
        first_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    
    // Header istatistikleri (header_name -> value -> count)
    $db->exec("CREATE TABLE IF NOT EXISTS header_stats (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        header_name TEXT,
        header_value TEXT,
        count INTEGER DEFAULT 1,
        first_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        UNIQUE(header_name, header_value)
    )");
    
    // Header sıralama pattern'leri
    $db->exec("CREATE TABLE IF NOT EXISTS header_order_patterns (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        pattern_hash TEXT UNIQUE,
        header_order TEXT,
        count INTEGER DEFAULT 1,
        first_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    
    // URL pattern'leri (query string analizi için)
    $db->exec("CREATE TABLE IF NOT EXISTS url_patterns (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        full_url TEXT,
        path TEXT,
        query_string TEXT,
        count INTEGER DEFAULT 1,
        first_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    
    // Referer analizi
    $db->exec("CREATE TABLE IF NOT EXISTS referers (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        referer TEXT UNIQUE,
        count INTEGER DEFAULT 1,
        first_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    
    // Raw request samples (son N istek detaylı)
    $db->exec("CREATE TABLE IF NOT EXISTS raw_requests (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
        ip_hash TEXT,
        method TEXT,
        full_url TEXT,
        user_agent TEXT,
        all_headers TEXT,
        header_order TEXT,
        query_params TEXT,
        protocol TEXT,
        connection_type TEXT
    )");
    
    // Request timing analizi (burst detection)
    $db->exec("CREATE TABLE IF NOT EXISTS request_timing (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        timestamp REAL,
        ip_hash TEXT,
        user_agent_hash TEXT
    )");
    
    // İndeksler
    $db->exec("CREATE INDEX IF NOT EXISTS idx_timing_ts ON request_timing(timestamp)");
    $db->exec("CREATE INDEX IF NOT EXISTS idx_timing_ip ON request_timing(ip_hash)");
}

/**
 * Gelen isteği kaydet
 */
function collectRequest($db) {
    $timestamp = microtime(true);
    $now = date('Y-m-d H:i:s');
    
    // IP hash (privacy için)
    $ipHash = md5($_SERVER['REMOTE_ADDR'] ?? 'unknown');
    
    // Tüm header'ları al
    $allHeaders = [];
    $headerOrder = [];
    
    foreach ($_SERVER as $key => $value) {
        if (strpos($key, 'HTTP_') === 0) {
            $headerName = str_replace('_', '-', substr($key, 5));
            $allHeaders[$headerName] = $value;
            $headerOrder[] = $headerName;
        }
    }
    
    // Özel header'lar
    if (isset($_SERVER['CONTENT_TYPE'])) {
        $allHeaders['Content-Type'] = $_SERVER['CONTENT_TYPE'];
    }
    if (isset($_SERVER['CONTENT_LENGTH'])) {
        $allHeaders['Content-Length'] = $_SERVER['CONTENT_LENGTH'];
    }
    
    // User-Agent
    $userAgent = $allHeaders['USER-AGENT'] ?? 'Not Set';
    
    // Full URL
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'unknown';
    $uri = $_SERVER['REQUEST_URI'] ?? '/';
    $fullUrl = $protocol . '://' . $host . $uri;
    
    // Query params
    $queryString = $_SERVER['QUERY_STRING'] ?? '';
    $queryParams = [];
    parse_str($queryString, $queryParams);
    
    // Referer
    $referer = $allHeaders['REFERER'] ?? 'Direct';
    
    // Connection type
    $connection = $allHeaders['CONNECTION'] ?? 'Not Set';
    
    // Request method
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    
    // HTTP Protocol version
    $httpProtocol = $_SERVER['SERVER_PROTOCOL'] ?? 'HTTP/1.1';
    
    // ===== 1. User-Agent istatistiği =====
    $stmt = $db->prepare("INSERT INTO user_agents (user_agent, count, first_seen, last_seen) 
                          VALUES (?, 1, ?, ?)
                          ON CONFLICT(user_agent) DO UPDATE SET 
                          count = count + 1, last_seen = ?");
    $stmt->execute([$userAgent, $now, $now, $now]);
    
    // ===== 2. Her header için istatistik =====
    $headerStmt = $db->prepare("INSERT INTO header_stats (header_name, header_value, count, first_seen, last_seen)
                                VALUES (?, ?, 1, ?, ?)
                                ON CONFLICT(header_name, header_value) DO UPDATE SET
                                count = count + 1, last_seen = ?");
    
    foreach ($allHeaders as $name => $value) {
        // Çok uzun değerleri kısalt
        $truncatedValue = substr($value, 0, 500);
        $headerStmt->execute([$name, $truncatedValue, $now, $now, $now]);
    }
    
    // ===== 3. Header sıralama pattern'i =====
    $orderString = implode(' | ', $headerOrder);
    $orderHash = md5($orderString);
    
    $stmt = $db->prepare("INSERT INTO header_order_patterns (pattern_hash, header_order, count, first_seen, last_seen)
                          VALUES (?, ?, 1, ?, ?)
                          ON CONFLICT(pattern_hash) DO UPDATE SET
                          count = count + 1, last_seen = ?");
    $stmt->execute([$orderHash, $orderString, $now, $now, $now]);
    
    // ===== 4. URL pattern =====
    $path = parse_url($uri, PHP_URL_PATH) ?: '/';
    $stmt = $db->prepare("INSERT INTO url_patterns (full_url, path, query_string, count, first_seen, last_seen)
                          VALUES (?, ?, ?, 1, ?, ?)
                          ON CONFLICT(full_url) DO UPDATE SET
                          count = count + 1, last_seen = ?");
    // URL'leri benzersiz yaparak duplicat'ları önle ama çok fazla farklı URL varsa sadece path bazlı grupla
    $urlKey = $path . '?' . $queryString;
    $stmt->execute([$urlKey, $path, $queryString, $now, $now, $now]);
    
    // ===== 5. Referer istatistiği =====
    $stmt = $db->prepare("INSERT INTO referers (referer, count, first_seen, last_seen)
                          VALUES (?, 1, ?, ?)
                          ON CONFLICT(referer) DO UPDATE SET
                          count = count + 1, last_seen = ?");
    $stmt->execute([$referer, $now, $now, $now]);
    
    // ===== 6. Raw request sample (son N kayıt) =====
    $stmt = $db->prepare("INSERT INTO raw_requests 
                          (timestamp, ip_hash, method, full_url, user_agent, all_headers, header_order, query_params, protocol, connection_type)
                          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $now,
        $ipHash,
        $method,
        $fullUrl,
        $userAgent,
        json_encode($allHeaders, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT),
        $orderString,
        json_encode($queryParams, JSON_UNESCAPED_UNICODE),
        $httpProtocol,
        $connection
    ]);
    
    // Eski raw kayıtları temizle
    $db->exec("DELETE FROM raw_requests WHERE id NOT IN (SELECT id FROM raw_requests ORDER BY id DESC LIMIT " . MAX_RAW_SAMPLES . ")");
    
    // ===== 7. Timing analizi =====
    if (ENABLE_TIMING_ANALYSIS) {
        $uaHash = md5($userAgent);
        $stmt = $db->prepare("INSERT INTO request_timing (timestamp, ip_hash, user_agent_hash) VALUES (?, ?, ?)");
        $stmt->execute([$timestamp, $ipHash, $uaHash]);
        
        // 1 saatten eski timing kayıtlarını temizle
        $oneHourAgo = $timestamp - 3600;
        $db->exec("DELETE FROM request_timing WHERE timestamp < $oneHourAgo");
    }
}
