// CYBERPUNK TRANSFORMATION - Complete JavaScript Overhaul

class CyberpunkInterface {
    constructor() {
        this.initializeMatrix();
        this.setupNavigationSystem();
        this.initializeParticleEngine();
        this.setupInteractiveElements();
        this.startSystemAnimations();
    }

    initializeMatrix() {
        this.matrixCanvas = document.getElementById('cyberCanvas');
        if (!this.matrixCanvas) return;
        
        this.matrixContext = this.matrixCanvas.getContext('2d');
        this.matrixParticles = [];
        this.matrixColumns = 0;
        this.matrixColumnWidth = 20;
        
        this.resizeMatrix();
        this.createMatrixParticles();
        this.animateMatrix();
        
        window.addEventListener('resize', () => this.resizeMatrix());
    }

    resizeMatrix() {
        this.matrixCanvas.width = window.innerWidth;
        this.matrixCanvas.height = window.innerHeight;
        this.matrixColumns = Math.floor(this.matrixCanvas.width / this.matrixColumnWidth);
    }

    createMatrixParticles() {
        this.matrixParticles = [];
        for (let i = 0; i < this.matrixColumns; i++) {
            this.matrixParticles.push({
                x: i * this.matrixColumnWidth,
                y: Math.random() * this.matrixCanvas.height,
                speed: Math.random() * 3 + 1,
                opacity: Math.random(),
                char: this.getRandomChar()
            });
        }
    }

    getRandomChar() {
        const chars = '01アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワヲン';
        return chars[Math.floor(Math.random() * chars.length)];
    }

    animateMatrix() {
        this.matrixContext.fillStyle = 'rgba(10, 10, 10, 0.05)';
        this.matrixContext.fillRect(0, 0, this.matrixCanvas.width, this.matrixCanvas.height);
        
        this.matrixContext.fillStyle = '#00ffff';
        this.matrixContext.font = '15px monospace';
        
        this.matrixParticles.forEach(particle => {
            this.matrixContext.globalAlpha = particle.opacity;
            this.matrixContext.fillText(particle.char, particle.x, particle.y);
            
            particle.y += particle.speed;
            particle.opacity -= 0.01;
            
            if (particle.y > this.matrixCanvas.height || particle.opacity <= 0) {
                particle.y = 0;
                particle.opacity = 1;
                particle.char = this.getRandomChar();
            }
        });
        
        this.matrixContext.globalAlpha = 1;
        requestAnimationFrame(() => this.animateMatrix());
    }

    setupNavigationSystem() {
        this.menuToggle = document.getElementById('cyberMenuToggle');
        this.menuOverlay = document.getElementById('cyberMenuOverlay');
        this.menuClose = document.getElementById('cyberMenuClose');
        
        if (this.menuToggle && this.menuOverlay) {
            this.menuToggle.addEventListener('click', () => this.toggleMenu());
        }
        
        if (this.menuClose && this.menuOverlay) {
            this.menuClose.addEventListener('click', () => this.closeMenu());
        }
        
        if (this.menuOverlay) {
            this.menuOverlay.addEventListener('click', (e) => {
                if (e.target === this.menuOverlay) {
                    this.closeMenu();
                }
            });
        }
    }

    toggleMenu() {
        this.menuOverlay.classList.toggle('active');
        document.body.style.overflow = this.menuOverlay.classList.contains('active') ? 'hidden' : 'auto';
    }

    closeMenu() {
        this.menuOverlay.classList.remove('active');
        document.body.style.overflow = 'auto';
    }

    initializeParticleEngine() {
        this.particleSystems = [];
        this.createFloatingParticles();
    }

    createFloatingParticles() {
        const particleContainer = document.createElement('div');
        particleContainer.className = 'cyber-particles';
        particleContainer.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 1;
        `;
        document.body.appendChild(particleContainer);
        
        for (let i = 0; i < 50; i++) {
            this.createFloatingParticle(particleContainer);
        }
    }

    createFloatingParticle(container) {
        const particle = document.createElement('div');
        particle.style.cssText = `
            position: absolute;
            width: 2px;
            height: 2px;
            background: #00ffff;
            border-radius: 50%;
            box-shadow: 0 0 6px #00ffff;
            animation: float ${Math.random() * 10 + 10}s linear infinite;
        `;
        
        particle.style.left = Math.random() * 100 + '%';
        particle.style.animationDelay = Math.random() * 10 + 's';
        
        container.appendChild(particle);
        
        // Add floating animation
        if (!document.getElementById('floatAnimation')) {
            const style = document.createElement('style');
            style.id = 'floatAnimation';
            style.textContent = `
                @keyframes float {
                    0% { transform: translateY(100vh) rotate(0deg); opacity: 0; }
                    10% { opacity: 1; }
                    90% { opacity: 1; }
                    100% { transform: translateY(-100px) rotate(360deg); opacity: 0; }
                }
            `;
            document.head.appendChild(style);
        }
    }

    setupInteractiveElements() {
        this.setupButtonInteractions();
        this.setupCardAnimations();
        this.setupScrollEffects();
    }

    setupButtonInteractions() {
        const cyberButtons = document.querySelectorAll('.cyber-btn, .cyber-cta');
        cyberButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                this.createRippleEffect(e);
                this.playButtonSound();
            });
        });
    }

    createRippleEffect(event) {
        const button = event.currentTarget;
        const ripple = document.createElement('span');
        const rect = button.getBoundingClientRect();
        const size = Math.max(rect.width, rect.height);
        const x = event.clientX - rect.left - size / 2;
        const y = event.clientY - rect.top - size / 2;
        
        ripple.style.cssText = `
            position: absolute;
            width: ${size}px;
            height: ${size}px;
            left: ${x}px;
            top: ${y}px;
            background: rgba(0, 255, 255, 0.3);
            border-radius: 50%;
            transform: scale(0);
            animation: ripple 0.6s linear;
            pointer-events: none;
        `;
        
        button.style.position = 'relative';
        button.style.overflow = 'hidden';
        button.appendChild(ripple);
        
        setTimeout(() => ripple.remove(), 600);
        
        if (!document.getElementById('rippleAnimation')) {
            const style = document.createElement('style');
            style.id = 'rippleAnimation';
            style.textContent = `
                @keyframes ripple {
                    to { transform: scale(4); opacity: 0; }
                }
            `;
            document.head.appendChild(style);
        }
    }

    playButtonSound() {
        // Create a simple beep sound using Web Audio API
        if (typeof AudioContext !== 'undefined' || typeof webkitAudioContext !== 'undefined') {
            const audioContext = new (AudioContext || webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
            oscillator.type = 'square';
            
            gainNode.gain.setValueAtTime(0.1, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.1);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.1);
        }
    }

    setupCardAnimations() {
        const cards = document.querySelectorAll('.cyber-card');
        cards.forEach(card => {
            card.addEventListener('mouseenter', () => {
                this.animateCard(card, 'enter');
            });
            
            card.addEventListener('mouseleave', () => {
                this.animateCard(card, 'leave');
            });
        });
    }

    animateCard(card, action) {
        const glow = card.querySelector('.cyber-card__glow');
        if (glow) {
            if (action === 'enter') {
                glow.style.animation = 'pulse 1s infinite';
            } else {
                glow.style.animation = 'pulse 2s infinite';
            }
        }
    }

    setupScrollEffects() {
        this.setupParallaxScrolling();
        this.setupScrollReveal();
    }

    setupParallaxScrolling() {
        window.addEventListener('scroll', () => {
            const scrolled = window.pageYOffset;
            const parallaxElements = document.querySelectorAll('.cyber-hero__content');
            
            parallaxElements.forEach(element => {
                const speed = 0.5;
                element.style.transform = `translateY(${scrolled * speed}px)`;
            });
        });
    }

    setupScrollReveal() {
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.style.opacity = '1';
                    entry.target.style.transform = 'translateY(0)';
                }
            });
        }, observerOptions);
        
        const revealElements = document.querySelectorAll('.cyber-card, .cyber-info-card');
        revealElements.forEach(element => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(50px)';
            element.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
            observer.observe(element);
        });
    }

    startSystemAnimations() {
        this.animateGlitchText();
        this.animateScanlines();
        this.startSystemPulse();
    }

    animateGlitchText() {
        const glitchElements = document.querySelectorAll('.cyber-hero__glitch');
        glitchElements.forEach(element => {
            setInterval(() => {
                element.style.textShadow = `
                    ${Math.random() * 4 - 2}px ${Math.random() * 4 - 2}px 0 #ff0000,
                    ${Math.random() * 4 - 2}px ${Math.random() * 4 - 2}px 0 #00ff00,
                    ${Math.random() * 4 - 2}px ${Math.random() * 4 - 2}px 0 #0000ff
                `;
                
                setTimeout(() => {
                    element.style.textShadow = '0 0 20px #00ffff';
                }, 100);
            }, 2000);
        });
    }

    animateScanlines() {
        const scanlines = document.querySelector('.cyber-hero__scanlines');
        if (scanlines) {
            let position = 0;
            setInterval(() => {
                position += 2;
                scanlines.style.backgroundPosition = `0 ${position}px`;
            }, 50);
        }
    }

    startSystemPulse() {
        const pulseElements = document.querySelectorAll('.cyber-card__glow, .cyber-section-title__line');
        pulseElements.forEach(element => {
            setInterval(() => {
                element.style.boxShadow = `
                    0 0 ${Math.random() * 20 + 10}px #00ffff,
                    0 0 ${Math.random() * 30 + 20}px #39ff14
                `;
            }, 1000);
        });
    }
}

// Initialize the cyberpunk interface when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new CyberpunkInterface();
});

// Additional utility functions
class CyberpunkUtils {
    static createGlitchEffect(element) {
        const originalText = element.textContent;
        const glitchChars = '!@#$%^&*()_+-=[]{}|;:,.<>?';
        
        setInterval(() => {
            let glitchedText = '';
            for (let i = 0; i < originalText.length; i++) {
                if (Math.random() < 0.1) {
                    glitchedText += glitchChars[Math.floor(Math.random() * glitchChars.length)];
                } else {
                    glitchedText += originalText[i];
                }
            }
            element.textContent = glitchedText;
            
            setTimeout(() => {
                element.textContent = originalText;
            }, 100);
        }, 3000);
    }
    
    static addHoverGlow(element) {
        element.addEventListener('mouseenter', () => {
            element.style.boxShadow = '0 0 30px #00ffff, 0 0 60px #39ff14';
        });
        
        element.addEventListener('mouseleave', () => {
            element.style.boxShadow = '';
        });
    }
    
    static createTypingEffect(element, text, speed = 100) {
        element.textContent = '';
        let i = 0;
        
        const typeInterval = setInterval(() => {
            element.textContent += text[i];
            i++;
            
            if (i >= text.length) {
                clearInterval(typeInterval);
            }
        }, speed);
    }
}

// Export for potential external use
window.CyberpunkInterface = CyberpunkInterface;
window.CyberpunkUtils = CyberpunkUtils;